using System;
using System.Collections.Generic;
using System.IO;
using Server;
using Server.Commands;
using Server.ContextMenus;
using Server.Items;
using Server.Engines.Lua;
using LuaInterface;

namespace Server.Mobiles
{
	public class LuaMobile : BaseCreature, ILuaCodeConsumer
	{
		private Lua m_Lua;
		private LuaCode m_Code;
		private string m_Script;

		public LuaInterface.Lua Lua
		{
			get 
            {
                if (m_Lua == null)
                    m_Lua = new LuaInterface.Lua();
                return m_Lua; 
            }
		}

		public LuaCode Code
		{
			get { return m_Code; }
			set
			{
				if ( m_Code != null )
				{
					m_Code.Unsubscribe( this );
				}

				m_Code = value;

				if ( m_Code != null )
				{
					m_Code.Subscribe( this );
				}
			}
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public string Script
		{
			get { return m_Script; }
			set
			{
				m_Script = value;

				Code = (LuaCode)LuaCodeBase.Scripts[m_Script];
			}
		}

		public string Description
		{
			get { return String.Format( "LuaMobile '{0}', Type '{1}', Serial {2}", this.Name, this.GetType().ToString(), this.Serial.Value.ToString() ); }
		}

		private void Update()
		{
			if ( m_Code != null )
			{
				m_Lua = new LuaInterface.Lua();

				try
				{
					m_Lua["this"] = this;

					m_Lua.DoString( m_Code.Code );

					LuaHelper.CallFunction( this, "OnInitialize" );
				}
				catch ( Exception e )
				{
					LuaHelper.BroudcastError( this, e.Message );
				}
			}
		}

		private static int GetRandomHue()
		{
			switch ( Utility.Random( 6 ) )
			{
				default:
				case 0:
					return 0;
				case 1:
					return Utility.RandomBlueHue();
				case 2:
					return Utility.RandomGreenHue();
				case 3:
					return Utility.RandomRedHue();
				case 4:
					return Utility.RandomYellowHue();
				case 5:
					return Utility.RandomNeutralHue();
			}
		}

		public void OnLostSubscription()
		{
			Script = null;
			Update();
		}

		public void OnScriptChanged()
		{
			Update();
		}

		[Constructable]
		public LuaMobile()
			: this( Utility.RandomBool() )
		{
		}

		[Constructable]
		public LuaMobile( bool female )
			: this( AIType.AI_Melee, FightMode.None, 10, 1, 0.8, 3.0 )
		{
			SetStr( 10, 30 );
			SetDex( 10, 30 );
			SetInt( 10, 30 );

			Fame = 50;
			Karma = 50;

			CanHearGhosts = true;

			SpeechHue = Utility.RandomDyedHue();
			Title = string.Empty;
			Hue = Utility.RandomSkinHue();
			Female = female;

			if ( Female )
			{
				Body = 0x191;
				Name = NameList.RandomName( "female" );
				HairItemID = Utility.RandomList( 0x203B, 0x203C, 0x203D, 0x2045, 0x204A, 0x2046, 0x2049 );
				HairHue = Utility.RandomHairHue();

				Item hat = null;

				switch ( Utility.Random( 5 ) )//4 hats, one empty, for no hat
				{
					case 0:
						hat = new FloppyHat( Utility.RandomNeutralHue() );
						break;
					case 1:
						hat = new FeatheredHat( Utility.RandomNeutralHue() );
						break;
					case 2:
						hat = new Bonnet();
						break;
					case 3:
						hat = new Cap( Utility.RandomNeutralHue() );
						break;
				}

				AddItem( hat );

				Item pants = null;

				switch ( Utility.Random( 3 ) )
				{
					case 0:
						pants = new ShortPants( GetRandomHue() );
						break;
					case 1:
						pants = new LongPants( GetRandomHue() );
						break;
					case 2:
						pants = new Skirt( GetRandomHue() );
						break;
				}

				AddItem( pants );

				Item shirt = null;

				switch ( Utility.Random( 7 ) )
				{
					case 0:
						shirt = new Doublet( GetRandomHue() );
						break;
					case 1:
						shirt = new Surcoat( GetRandomHue() );
						break;
					case 2:
						shirt = new Tunic( GetRandomHue() );
						break;
					case 3:
						shirt = new FancyDress( GetRandomHue() );
						break;
					case 4:
						shirt = new PlainDress( GetRandomHue() );
						break;
					case 5:
						shirt = new FancyShirt( GetRandomHue() );
						break;
					case 6:
						shirt = new Shirt( GetRandomHue() );
						break;
				}

				AddItem( shirt );
			}
			else
			{
				Body = 0x190;
				Name = NameList.RandomName( "male" );
				HairItemID = Utility.RandomList( 0x203B, 0x203C, 0x203D, 0x2044, 0x2045, 0x2047, 0x2048 );
				HairHue = Utility.RandomHairHue();
				FacialHairItemID = Utility.RandomList( 0x0000, 0x203E, 0x203F, 0x2040, 0x2041, 0x2067, 0x2068, 0x2069 );
				FacialHairHue = this.HairHue;

				Item hat = null;

				switch ( Utility.Random( 7 ) ) //6 hats, one empty, for no hat
				{
					case 0:
						hat = new SkullCap( GetRandomHue() );
						break;
					case 1:
						hat = new Bandana( GetRandomHue() );
						break;
					case 2:
						hat = new WideBrimHat();
						break;
					case 3:
						hat = new TallStrawHat( Utility.RandomNeutralHue() );
						break;
					case 4:
						hat = new StrawHat( Utility.RandomNeutralHue() );
						break;
					case 5:
						hat = new TricorneHat( Utility.RandomNeutralHue() );
						break;
				}

				AddItem( hat );

				Item pants = null;

				switch ( Utility.Random( 2 ) )
				{
					case 0:
						pants = new ShortPants( GetRandomHue() );
						break;
					case 1:
						pants = new LongPants( GetRandomHue() );
						break;
				}

				AddItem( pants );

				Item shirt = null;

				switch ( Utility.Random( 5 ) )
				{
					case 0:
						shirt = new Doublet( GetRandomHue() );
						break;
					case 1:
						shirt = new Surcoat( GetRandomHue() );
						break;
					case 2:
						shirt = new Tunic( GetRandomHue() );
						break;
					case 3:
						shirt = new FancyShirt( GetRandomHue() );
						break;
					case 4:
						shirt = new Shirt( GetRandomHue() );
						break;
				}

				AddItem( shirt );
			}

			Item feet = null;

			switch ( Utility.Random( 3 ) )
			{
				case 0:
					feet = new Boots( Utility.RandomNeutralHue() );
					break;
				case 1:
					feet = new Shoes( Utility.RandomNeutralHue() );
					break;
				case 2:
					feet = new Sandals( Utility.RandomNeutralHue() );
					break;
			}

			AddItem( feet );

			Container pack = new Backpack();

			pack.DropItem( new Gold( 0, 50 ) );

			pack.Movable = false;

			AddItem( pack );
		}

		[Constructable]
		public LuaMobile( AIType ai, FightMode mode, int iRangePerception, int iRangeFight, double dActiveSpeed, double dPassiveSpeed )
			: base( ai, mode, iRangePerception, iRangeFight, dActiveSpeed, dPassiveSpeed )
		{
		}

		public override bool HandlesOnSpeech( Mobile from )
		{
			if ( from.InRange( this.Location, 12 ) )
				return true;

			return base.HandlesOnSpeech( from );
		}

		// Events

		public override void OnActionBackoff()
		{
			LuaHelper.CallFunction( this, "OnActionBackoff" );
		}

		public override void OnActionCombat()
		{
			LuaHelper.CallFunction( this, "OnActionCombat" );
		}

		public override void OnActionFlee()
		{
			LuaHelper.CallFunction( this, "OnActionFlee" );
		}

		public override void OnActionGuard()
		{
			LuaHelper.CallFunction( this, "OnActionGuard" );
		}

		public override void OnActionInteract()
		{
			LuaHelper.CallFunction( this, "OnActionInteract" );
		}

		public override void OnActionWander()
		{
			LuaHelper.CallFunction( this, "OnActionWander" );
		}

		public override bool OnBeforeDeath()
		{
			return LuaHelper.CallFunction( this, "OnBeforeDeath" ).AsBoolean( 0, base.OnBeforeDeath() );
		}

		public override void OnCombatantChange()
		{
			LuaHelper.CallFunction( this, "OnCombatantChange" );
		}

		public override void OnCured( Mobile from, Poison oldPoison )
		{
			LuaHelper.CallFunction( this, "OnCured", from, oldPoison );
		}

		public override void OnDamage( int amount, Mobile from, bool willKill )
		{
			LuaHelper.CallFunction( this, "OnDamage", amount, from, willKill );
		}

		public override void OnDamagedBySpell( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnDamagedBySpell", from );
		}

		public override void OnDeath( Container c )
		{
			LuaHelper.CallFunction( this, "OnDeath", c );
		}

		public override void OnDoubleClick( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnDoubleClick", from );
		}

		public override bool OnDragDrop( Mobile from, Item dropped )
		{
			return LuaHelper.CallFunction( this, "OnDragDrop", from, dropped ).AsBoolean( 0, base.OnDragDrop( from, dropped ) );
		}

		public override void OnFailedCure( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnFailedCure", from );
		}

		public override void OnGaveMeleeAttack( Mobile defender )
		{
			LuaHelper.CallFunction( this, "OnGaveMeleeAttack", defender );
		}

		public override void OnGotMeleeAttack( Mobile attacker )
		{
			LuaHelper.CallFunction( this, "OnGotMeleeAttack", attacker );
		}

// 		public override void OnHeal( ref int amount, Mobile from )
// 		{
// 			amount = LuaHelper.CallFunction( this, "OnHeal", amount, from ).AsInteger( 0, amount );
// 		}

		public override void OnHitsChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnHitsChange", oldValue );
		}

		public override void OnItemLifted( Mobile from, Item item )
		{
			LuaHelper.CallFunction( this, "OnItemLifted", from, item );
		}

		public override void OnKilledBy( Mobile mob )
		{
			LuaHelper.CallFunction( this, "OnKilledBy", mob );
		}

		public override void OnKillsChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnKillsChange", oldValue );
		}

		public override void OnManaChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnManaChange", oldValue );
		}

		protected override void OnLocationChange( Point3D oldLocation )
		{
			LuaHelper.CallFunction( this, "OnLocationChange", oldLocation );
		}

		protected override void OnMapChange( Map oldMap )
		{
			LuaHelper.CallFunction( this, "OnMapChange", oldMap );
		}

		public override void OnPoisoned( Mobile from, Poison poison, Poison oldPoison )
		{
			LuaHelper.CallFunction( this, "OnPoisoned", from, poison, oldPoison );
		}

		public override bool OnMoveOff( Mobile m )
		{
			return LuaHelper.CallFunction( this, "OnMoveOff", m ).AsBoolean( 0, base.OnMoveOff( m ) );
		}

		public override bool OnMoveOver( Mobile m )
		{
			return LuaHelper.CallFunction( this, "OnMoveOver", m ).AsBoolean( 0, base.OnMoveOver( m ) );
		}

		public override void OnMovement( Mobile m, Point3D oldLocation )
		{
			LuaHelper.CallFunction( this, "OnMovement", m, oldLocation );
		}

		public override void OnSingleClick( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnSingleClick", from );
		}

		public override void OnSkillInvalidated( Skill skill )
		{
			LuaHelper.CallFunction( this, "OnSkillInvalidated", skill );
		}

		public override void OnSpeech( SpeechEventArgs e )
		{
			LuaHelper.CallFunction( this, "OnSpeech", e );
		}

		public override void OnSpellCast( ISpell spell )
		{
			LuaHelper.CallFunction( this, "OnSpellCast", spell );
		}

		public override void OnStamChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnStamChange", oldValue );
		}

		public override void OnThink()
		{
			LuaHelper.CallFunction( this, "OnThink" );
		}

		public override void OnFameChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnFameChange", oldValue );
		}

		public override void OnKarmaChange( int oldValue )
		{
			LuaHelper.CallFunction( this, "OnKarmaChange", oldValue );
		}

		public override void GetContextMenuEntries( Mobile from, List<ContextMenuEntry> list )
		{
			if ( m_Lua.GetFunction( "OnTalk" ) != null )
			{
				list.Add( new TalkEntry( this ) );
			}

			base.GetContextMenuEntries( from, list );
		}

		public LuaMobile( Serial serial )
			: base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int)1 ); // version

			writer.Write( (string)m_Script );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
					{
						Script = reader.ReadString();

						break;
					}
			}
		}

		private class TalkEntry : ContextMenuEntry
		{
			private LuaMobile m_Mobile;

			public TalkEntry( LuaMobile mobile )
				: base( 6146 )
			{
				m_Mobile = mobile;
			}

			public override void OnClick()
			{
				Mobile from = Owner.From;

				LuaHelper.CallFunction( m_Mobile, "OnTalk", from );
			}
		}
	}
}